// --- START OF FILE content.js ---

(function () {
  // בדיקה: האם הסקריפט כבר הוגדר בעבר בדף הזה?
  if (!window.keyboardFixerEngine) {

    // --- הגדרות ---

    const engToHeb = {
      q: 'ת', w: "'", e: 'ק', r: 'ר', t: 'א', y: 'ט', u: 'ו', i: 'ן', o: 'ם', p: 'פ',
      a: 'ש', s: 'ד', d: 'ג', f: 'כ', g: 'ע', h: 'י', j: 'ח', k: 'ל', l: 'ך',
      z: 'ז', x: 'ס', c: 'ב', v: 'ה', b: 'נ', n: 'מ', m: 'צ',
      ',': 'ת', '/': 'ץ', ';': 'ף',
      Q: 'ת', W: "'", E: 'ק', R: 'ר', T: 'א', Y: 'ט', U: 'ו', I: 'ן', O: 'ם', P: 'פ',
      A: 'ש', S: 'ד', D: 'ג', F: 'כ', G: 'ע', H: 'י', J: 'ח', K: 'ל', L: 'ך',
      Z: 'ז', X: 'ס', C: 'ב', V: 'ה', B: 'נ', N: 'מ', M: 'צ'
    };

    const hebToEng = {
      'ת': ',', 'ץ': '/', 'ף': ';',
      '/': 'q', "'": 'w', 'ק': 'e', 'ר': 'r', 'א': 't', 'ט': 'y', 'ו': 'u', 'ן': 'i', 'ם': 'o', 'פ': 'p',
      'ש': 'a', 'ד': 's', 'ג': 'd', 'כ': 'f', 'ע': 'g', 'י': 'h', 'ח': 'j', 'ל': 'k', 'ך': 'l',
      'ז': 'z', 'ס': 'x', 'ב': 'c', 'ה': 'v', 'נ': 'b', 'מ': 'n', 'צ': 'm'
    };

    function countHebrew(text) {
      return (text.match(/[\u0590-\u05FF]/g) || []).length;
    }
    function countEnglish(text) {
      return (text.match(/[a-zA-Z,\/;]/g) || []).length;
    }
    function detectLanguage(text) {
      const hebCount = countHebrew(text);
      const engCount = countEnglish(text);
      if (hebCount === 0 && engCount === 0) return 'none';
      return hebCount > engCount ? 'he' : 'en';
    }
    function isUrl(word) {
      const lower = word.toLowerCase().trim();
      const urlPattern = /^(https?:\/\/|www\.)|(\.(com|net|org|gov|edu|il|co|io|me|biz|info|tv|us|uk|ru|de|fr|es|cn|jp|au))$/i;
      return urlPattern.test(lower);
    }

    function convertWord(word, lang) {
      if (isUrl(word)) return word;
      if (lang === 'he') return [...word].map(c => hebToEng[c] || c).join('');
      if (lang === 'en') return [...word].map(c => engToHeb[c] || c).join('');
      return word;
    }

    function convertTextAuto(text) {
      const lang = detectLanguage(text);
      return text.split(/(\s+)/).map(w => convertWord(w, lang)).join('');
    }

    // פונקציה קריטית: מודיעה לאתר שהטקסט השתנה
    function triggerEvents(element) {
      // רשימת אירועים שאתרים בדרך כלל מאזינים להם
      const events = ['input', 'change', 'keydown', 'keyup'];

      events.forEach(eventType => {
        const event = new Event(eventType, { bubbles: true, cancelable: true });
        element.dispatchEvent(event);
      });
    }

    function runConversion() {
      const el = document.activeElement;
      if (!el) return;

      // 1. טיפול ב-Input / Textarea רגילים
      if (el.tagName === 'INPUT' || el.tagName === 'TEXTAREA') {
        const start = el.selectionStart;
        const end = el.selectionEnd;
        const newValue = convertTextAuto(el.value);

        // טריק מיוחד לאתרים מבוססי React (כמו פייסבוק/אינסטגרם)
        // לפעמים סתם el.value = x לא עובד להם, צריך לקרוא ל-setter המקורי
        const nativeInputValueSetter = Object.getOwnPropertyDescriptor(window.HTMLInputElement.prototype, "value").set;
        const nativeTextAreaSetter = Object.getOwnPropertyDescriptor(window.HTMLTextAreaElement.prototype, "value").set;

        if (el.tagName === 'INPUT' && nativeInputValueSetter) {
          nativeInputValueSetter.call(el, newValue);
        } else if (el.tagName === 'TEXTAREA' && nativeTextAreaSetter) {
          nativeTextAreaSetter.call(el, newValue);
        } else {
          el.value = newValue;
        }

        // הפעלת אירועים כדי שהאתר יתעורר
        triggerEvents(el);

        // שחזור סמן
        el.setSelectionRange(start, end);
      }

      // 2. טיפול ב-ContentEditable (וואטסאפ, ג'ימייל, אופיס אונליין)
      else if (el.isContentEditable) {
        const text = el.innerText || el.textContent;
        const converted = convertTextAuto(text);

        // בחירה של כל הטקסט
        document.execCommand('selectAll', false, null);

        // החלפה באמצעות insertText - זה הפתרון הכי טוב כי הדפדפן מחשיב את זה כהקלדה אמיתית
        if (!document.execCommand('insertText', false, converted)) {
          // גיבוי למקרה ש-execCommand נכשל
          el.textContent = converted;
        }

        // ליתר ביטחון, נפעיל אירועים גם כאן
        triggerEvents(el);
      }
    }

    window.keyboardFixerEngine = {
      run: runConversion
    };
  }

  // הפעלה
  window.keyboardFixerEngine.run();

})();